package ch.brx.ldifviewer;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import ch.brx.ldifviewer.TemplateReader.TemplateNameFields;

import com.unboundid.ldif.DuplicateValueBehavior;
import com.unboundid.ldif.LDIFException;
import com.unboundid.ldif.LDIFReader;
import com.unboundid.ldif.TrailingSpaceBehavior;
import com.unboundid.ldap.sdk.Attribute;
import com.unboundid.ldap.sdk.Entry;


public class ContactsLDIFReader extends ContactsReader  {

	NameSchema schema;
	String filename;
	ArrayList<ContactListEntry> list = null;
	Entry dnEntry;
	String dn;
	TemplateReader template;
	boolean load = false;

	public ContactsLDIFReader(String filename, NameSchema schema, TemplateReader template)
	{
		System.out.println("New ContactsLDIFReader: " + this.toString());

		this.filename = filename;
		this.schema = schema;
		this.template = template;
	}

	@Override
	public void setNameSchema(NameSchema schema)
	{
		unload();
		this.schema = schema;
	}

	@Override
	ArrayList<ContactListEntry> getList() throws IOException
	{
		if(list == null)
		{
			list = new ArrayList<ContactListEntry>();
			load(true, false);
		}

		return list;
	}

	@Override
	Map<String, List<String>> getEntry(String dn) throws IOException
	{
		Map<String, List<String>> e = null;

		if( !dn.equals(this.dn) )
		{
			this.dn = dn;
			dnEntry = null;
		}

		if(dnEntry == null)
		{
			load(false, true);
		}

		if(dnEntry != null)
		{
			e = new HashMap<String, List<String> >();
			java.util.Collection<Attribute> attributes = dnEntry.getAttributes();

			for(Attribute a: attributes){
				ArrayList<String> arr = new ArrayList<String>(); 
				for(String v: dnEntry.getAttributeValues(a.getName()) )
				{
					arr.add(v);
				}
				e.put(a.getName(), arr);
			}			
		}

		return e;
	}
	
	void cancel()
	{
		load = false;
	}

	private void unload()
	{
		this.list = null;
		this.dnEntry = null;
		this.dn = null;
	}

	private void load(boolean loadList, boolean loadEntry) throws IOException
	{
		if(loadList)
		{
			list.clear();
		}
		
		load = true;

		TemplateNameFields flds = template.getNameFields();
		if(flds == null)
		{
			throw new IOException("Template does not contain field names.");
		}

		List<String> searchfields = template.getSearchFields();

		try {
			LDIFReader ldifReader = new LDIFReader(filename);
			ldifReader.setTrailingSpaceBehavior(TrailingSpaceBehavior.STRIP);
			ldifReader.setDuplicateValueBehavior(DuplicateValueBehavior.RETAIN);

			while (load)
			{
				Entry entry;
				try
				{
					entry = ldifReader.readEntry();
					if (entry == null)
					{
						break;
					}
				}
				catch (LDIFException le)
				{
					if (le.mayContinueReading())
					{
						System.err.println("A recoverable occurred while attempting to " +
								"read an entry at or near line number " + le.getLineNumber() +
								":  " + le.getMessage());
						System.err.println("The entry will be skipped.");
						continue;
					}
					else
					{
						System.err.println("An unrecoverable occurred while attempting to " +
								"read an entry at or near line number " + le.getLineNumber() +
								":  " + le.getMessage());
						System.err.println("LDIF processing will be aborted.");
						break;
					}
				}
				catch (IOException ioe)
				{
					System.err.println("An I/O error occurred while attempting to read " +
							"from the LDIF file:  " + ioe.getMessage());
					System.err.println("LDIF processing will be aborted.");
					break;
				}

				if(loadList)
				{
					String name = null;
					String cname, fname, lname;

					switch(schema)
					{
					case USE_NAME_CALL: 
						cname = entry.getAttributeValue(flds.callName);
						if(cname != null)
						{
							name = cname;
						}
						break;

					case USE_NAME_FIRST_LAST:
						fname = entry.getAttributeValue(flds.firstName);
						lname = entry.getAttributeValue(flds.lastName);
						if(fname != null && lname != null)
						{
							name = fname + " " + lname;
						}
						break;

					case USE_NAME_LAST_FIRST:
						fname = entry.getAttributeValue(flds.firstName);
						lname = entry.getAttributeValue(flds.lastName);
						if(fname != null && lname != null)
						{
							name = lname + " " + fname;
						}
						break;
					}

					if(name == null)
					{
						fname = entry.getAttributeValue(flds.firstName);
						lname = entry.getAttributeValue(flds.lastName);
						cname = entry.getAttributeValue(flds.callName);
						if(cname != null)
						{
							name = cname;
						}
						else if(fname != null && lname != null)
						{
							name = fname + " " + lname;
						}
						else if(lname != null)
						{
							name = lname;
						}
						else if(fname != null)
						{
							name = fname;
						}
						else if(entry.getAttributeValue("mail") != null)
						{
							name = entry.getAttributeValue("mail");
						}
						else
						{
							name = entry.getAttributeValue("dn");
						}
					}

					if(name == null)
					{
						name = "--";
					}

					StringBuilder info = new StringBuilder(64);
					for(String sfield: searchfields)
					{
						String val = entry.getAttributeValue(sfield);
						if(val != null)
						{
							boolean isNumber = true;
							for(int cix = 0; cix < val.length(); cix++)
							{
								if(Character.isLetter(val.charAt(cix)))
								{
									isNumber = false;
									break;
								}
							}

							if(isNumber)
							{
								info.append(" ");
								info.append(val.replace(" ", ""));
							}
							else
							{
								//add every word of sarr if not yet existing in name or info string
								String words[] = val.split(" ");
								for(String w:words)
								{
									if(name.indexOf(w) < 0 && info.indexOf(w) < 0)
									{
										info.append(" ");
										info.append(w);
									}
								}
							}

						}
					}

					list.add(new ContactListEntry(entry.getDN(),  name,  info.toString()));
				}
				else if(loadEntry)
				{
					if( dn.equals(entry.getDN()) )
					{
						dnEntry = entry;
						break;
					}
				}

			}

			ldifReader.close();
		}
		catch (IOException ioe)
		{
			System.err.println("An I/O error occurred while attempting to read " +
					"from the LDIF file:  " + ioe.getMessage());
			System.err.println("LDIF processing will be aborted.");
			throw ioe;
		}

		if(loadList)
		{
			Collections.sort(list);
		}
	}
}
